library('methods')
library('uuid')
library('httr')
library('Matrix')

adrVersion <- function() '2.0.264' 

logMsgPrefix <- function() 'ADR: ' 

logMessage <- function(tag, lev, msg) {
    logLev <- getADROption('loglevel', 0)
    taggedLogLevs <- getADROption('loglevels', list())
    tagLev <- taggedLogLevs[[tag]]
#    print(paste('logLev: tag', tag, 'msg lev', lev, 'global', logLev, ' list ',
#                paste(names(taggedLogLevs), taggedLogLevs, sep='=', collapse=', '), 'tagLev', tagLev))
    if (!is.null(tagLev)) {
        logLev <- tagLev
    }
    if (lev <= logLev) {
        if (getADROption('logmsg', TRUE)) {
            message(logMsgPrefix(), msg)
        } else {
#            cat(paste(logMsgPrefix(), Sys.time(), ' ', sprintf('%-28s', paste0(tag, ':')), ' ',
#                      msg, '\n', collapse='', sep=''))
            cat(paste0(logMsgPrefix(), msg, '\n'))
        }
    }
}

logMessageMsg <- function(tag, lev, msg) {
    message(msg)
}


globalADROptions <- local({ opts <- list(); function(name, val) {
    if (!missing(val)) {
        pval = if (is.environment(val)) { 'ENV' } else { val }
        logMessage('globalADROptions:set', 1, paste('Set global ADR option: ', name, ' = ', pval))
        opts[[name]] <<- val
    }
#    show(list(get.opt=name,v=opts[[name]]))
    opts[[name]]
}})

adrSetOption <- function(name, val) {
    old <- globalADROptions(name)
    new <- globalADROptions(name, val)
    attr(new, 'old') <- old
    new
}

adrGetOption <- function(name, def) {
    if (missing(def)) {
        def <- NULL
    }
    oval <- globalADROptions(name)
    if (is.null(oval)) {
        oval <- def
    }
    oval
}
getADROption <- function(...) {
    adrGetOption(...)
}

adrSetLoglev <- function(...) {
    args <- list(...)
    cl <- adrGetOption('loglevels', list())
    if (length(args) == 0) {
        return(cl);
    }
    n <- names(args)
    if (is.null(n) || length(n) == 0) {
        logMessage('adrSetLoglev:fail:noNames', logLError, paste('No names in loglevels list'))
        stop('no names')
        return(0)
    }
    for (k in 1:length(n)) {
        tag <- n[[k]]
        lev <- args[[tag]]
        cl[[tag]] <- NULL
        cl[[tag]] <- lev
    }
    adrSetOption('loglevels', cl)
}


adrGlobalOptions <- function(...) {
    args = list(...)
    if (length(args) == 0) return(NULL)
    if (is.null(names(args[1])) || nchar(names(args[1])) == 0) {
        if (is.character(args[[1]])) {
                                        # get
            res <- adrGetOption(args[[1]])
            return(res)
        }
    }
    res <- lapply(names(args), function(n) {
        globalADROptions(n, args[[n]])
    })
    res
}

adrGlobalOptions(ndd = 1, 'taylor-order' = 1)
globalADROptions('fkeymode', 'source')
globalADROptions('vectormode-switchover', 10)
globalADROptions('fdStep', 1e-8)
globalADROptions('independents', 1)
globalADROptions('vectormode', c())
#globalADROptions('vectorclass', 'advec')
globalADROptions('addirmode', 'fm')
globalNDD <- function(...) {
    do.call(globalADROptions, c('ndd', list(...)))
}
globalNDD(1)
globalADMode <- function(...) {
    do.call(globalADROptions, c('current.nddadmode', list(...)))
}
globalADMode(1)

adrGlobalNDD <- globalNDD
adrGlobalADMode <- globalADMode

logLDebug <- 10
logLChat <- 2
logLInfo <- 1
logLNote <- 0
logLWarn <- -1
logLError <- -2

d_zeros_scalar <- function(x) {
    if (is.list(x)) {
        r <- lapply(x, function(e) {
            d_zeros_scalar(e)
        })
        return (r)
    }
                                        # scalar mode
    if (is.array(x)) {
        r <- array(0, dim(x))
    } else if (is.vector(x)) {
        r <- vector("numeric", length(x))
    } else {
        r <- 0
    }
}

d_zeros_vector <- function(x) {
    if (is.list(x)) {
        r <- lapply(x, function(e) {
            d_zeros_vector(e)
        })
        return (r)
    }
                                        # vector mode
    r <- advec(x, globalNDD())
}


d_zeros <- function(x) {
    if (globalADMode() == 0) {
        d_zeros_scalar(x)
    } else {
        d_zeros_vector(x)
    }
}

totalNumel <- function(...) {
    args = list(...);
    nels = lapply(args, function(x) numel(x));
    tnel = do.call('sum', nels);
}

createZeroGradients <- function(...) {
    args = list(...);
    tnel = totalNumel(...);
    globalNDD(tnel);
    res = lapply(args, d_zeros);
    res
}

createSeededGradientsFor_advec <- function(S,...) {
    args = list(...);
    res = lapply(args, d_zeros);
    if (length(args) > 1) {
        offs = 1;
        for (i in 1:length(args)) {
            neli = numel(args[[i]]);
#            show(i)
#            show('neli')
#            show(neli)
#            show(safedim(args[[i]]))
#            show(class(args[[i]]))
#            show(res[[i]])
            res[[i]]@data <- as.array(S[offs:(offs+neli-1),])
            dim(res[[i]]@data) <- c(safedim(args[[i]]), res[[i]]@ndd);
#            show(dim(res[[i]]@data))
            offs <- offs + neli;
        }
    } else {
        res[[1]]@data <- S;
        dim(res[[1]]@data) <- c(safedim(args[[1]]), res[[1]]@ndd)
    }
#    show(res)
    res
}

createSeededGradientsFor <- function(S,..., index = 1) {
    args = list(...)
    ndd = dim(as.matrix(S))[2]
    if (length(args) == 0) {
        return(args)
    }
    if (!is.matrix(S) && !is.numeric(S)) {
        stop('adr:seed_not_matrix');
    }
    if (globalADMode() == 1) {
        globalNDD(ndd)
        res <- createSeededGradientsFor_advec(S, ...)
    } else {
        res <- args
        offs <- 1
        for (k in 1:length(args)) {
            nk <- numel(res[[k]])
            Sv <- if (length(safedim(S))>1) S[offs:(offs+nk-1), index] else S[offs:(offs+nk-1)]
            res[[k]] <- Sv
            dim(res[[k]]) <- dim(args[[k]])
            offs <- offs + nk
        }
    }
    res
}

createFullGradients <- function(..., index = 1) {
    tnel = totalNumel(...)
    createSeededGradientsFor(diag(tnel), ..., index = index)
}

jacFor <- function(...) {
    args = list(...);
    listp <- sapply(args, is.list)
    for (i in which(listp)) {
        args[[i]] <- do.call(jacFor, args[[i]])
    }
    if (is.advec(args[[1]])) {
        ndd = ndd_advec(args[[1]]);
#        show(ndd)
        if (length(args) > 1) {
            nx = totalNumel(...);
#            show(nx)
            Jac = matrix(array(0, c(nx, ndd)), ncol = ndd);
            offs = 1;
            for (i in 1:length(args)) {
                nix = numel(args[[i]]);
                Jac[offs:(offs+nix-1),] <- matrix(args[[i]]@data, ncol = ndd);
                offs <- offs + nix;
            }
        } else {
            Jac <- matrix(args[[1]]@data, ncol = ndd);
        }
    } else {
        Jac <- array(0, c(0,1))
        for (i in 1:length(args)) {
            Jp <- as.array(args[[i]])
            dim(Jp) <- c(prod(safedim(Jp)), 1)
            Jac <- rbind(Jac, Jp)
        }
    }
    Jac
}

getSource <- function(handle) {
    attributes(handle) <- c()
    r <- deparse(handle)
    r
}

adrGetSource <- function(handle) {
    getSource(handle)
}

toFile <- function(fname, txt) {
    fileConn<-file(fname)
    writeLines(txt, fileConn)
    close(fileConn)
}

fromFile <- function(fname) {
    fileConn<-file(fname)
    txt <- readLines(fileConn, -1)
    close(fileConn)
    txt
}

executeSystemCommand <- function(program, arguments) {
    print(paste("executeSystemCommand(", program, ' ', paste(arguments, collapse=' '),")", collapse=' ',sep=''), quote=FALSE)
    out <- system2(program, arguments, stdout=TRUE, stderr=TRUE, wait=TRUE)
#    print(paste("Command status ", attr(out, 'status'), ", errmsg: ", attr(out, 'errmsg'), ", output: '", out, "'"))
}

transformFile.OPXClient <- function(inFile, outFile, mode="r-ad-fm", params) {
    pargs <- paste(lapply(names(params), function(pname) {
        pval = params[[pname]]
        paste('-s ', pname, '=', paste(pval,sep='',collapse=','),collapse='',sep='')
    }),collapse=' ',sep='')
    print(paste("transformFile(", mode, "", pargs, "): ", inFile, " => ", outFile))
    executeSystemCommand("opx-client", c("-m", mode, pargs, "-o", outFile, inFile))
}

transformFile.RCurl <- function(inFile, outFile, mode="r-ad-fm", params) {

    params[['mode']] = mode
    params[['output']] = 'xml'
    params[['method-tcproc']] = ''
    
#    show(params)
    
    print(paste("transformFile(POST, ", paste(params, collapse='', sep=''), "): ", inFile, " => ", outFile))

    fdata <- paste(fromFile(inFile), collapse='\n', sep='')
#    fdata <- upload_file(normalizePath(inFile))
    
    r <- POST(adrServerURL(),
              transformText.Headers(),
              body = append(params, list(data1 = fdata)),
              encode = 'multipart')

    stop_for_status(r)

    data <- content(r, "text", encoding='utf-8')

    toFile(outFile, data)
    
}

transformFile <- function(...) {
    l <- list(...)
    do.call(transformFile.RCurl, l)
}

transformFromFile <- function(inputFile, ...) {
    l <- list(...)
    tmpFile = 'tmp.R'
    do.call(transformFile.RCurl, append(list(inputFile, tmpFile), l))
    res <- fromFile(tmpFile)
    #delete(tmpFile)
    res
}

diffFile <- function(inFile, outFile, mode="fm", actlist) {
    tc <- paste(c("r-ad", "-fm"), collapse="", sep="")
    plist <- list(mode = mode)
    if (length(actlist)) {
        plist$independents = actlist
    }
    transformFile(inFile, outFile, tc, params = plist)
#    print(paste("Differentiated code in: ", inFile, " to ", outFile, sep="", collapse=""), quote=FALSE);
#    print("Original code: ")
#    print(paste(fromFile(inFile), sep=''), quote=FALSE)
#    print("Diff. code: ")
#    print(paste(fromFile(outFile), sep=''), quote=FALSE)
}

transformText.TempFiles <- function(inText, mode="r-ad-fm", params = list()) {
    inFile <- 'tmp.txt';
    outFile <- 'tmp_out.txt';
    toFile(inFile, inText)
    params$mode <- mode
    transformFile(inFile, outFile, mode, params = params)
    r <- fromFile(outFile)
    r
}

toKeyValueParams <- function(params) {
    r <- list()
    i <- 0
    for (k in names(params)) {
        r[[paste('pname',i,sep='')]] <- k
        r[[paste('pvalue',i,sep='')]] <- params[[k]]
        i <- i + 1
    }
    r
}

adrServerURLDefault <- function() {
    serverURL <- "https://r-adr.de/"
    serverURL
}

adrServerURL <- function() {
    serverURL <- adrServerURLDefault()
    envServ <- Sys.getenv('ADR_SERVER')
    optServ <- getADROption('adrserver', envServ)
    if (!is.null(optServ) && nchar(optServ)) {
        serverURL <- optServ
    }
    serverURL
}
adrTransformURL <- function() {
    serverURL <- adrServerURL()
    serverURLPostFix <- adrGetOption('adrserver-path', 'adr/')
    if (!endsWith(serverURL, serverURLPostFix)) {
        sep = if (endsWith(serverURL, '/')) '' else '/'
        serverURL <- paste(serverURL, serverURLPostFix, sep=sep, collapse='')
    }
    serverURL
}

adrLookupLegalDocsDate <- function() {
    date <- Sys.Date()
    globalADROptions('LegalDocsDate', date)
    globalADROptions('LegalDocsDateLookup', Sys.Date())
    date
}

adrGetLegalDocsDate <- function() {
    date <- getADROption('LegalDocsDate')
    if (is.null(date)) {
        date <- adrLookupLegalDocsDate()
    } else {
        dateLookup <- getADROption('LegalDocsDateLookup')
        if (dateLookup + 10 < Sys.Date()) {
            date <- adrLookupLegalDocsDate()
        }
    }
    date
}

adrGetUserConfigDir <- function() {
    dname <- Sys.glob('~/.adr')
    if (length(dname) == 0) {
        dfullname <- paste(Sys.glob('~'), '.adr', sep='/', collapse='')
        dir.create(dfullname)
        dname <- Sys.glob('~/.adr')
        if (length(dname) == 0) {
            stop(paste('Failed to create user config directory', dfullname))
        }
    }
    dname
}

adrGetUserConfigFile <- function(name) {
    dfullname <- paste(adrGetUserConfigDir(), name, sep='/', collapse='')
}

adrGetAcceptLegalDocsStorage <- function() {
    fname <- adrGetUserConfigFile('accept-terms-of-use-and-privacy-stmt.Rds')
}
adrLookupAcceptLegalDocs <- function() {
    fname <- adrGetAcceptLegalDocsStorage()
    acc <- FALSE
    tryCatch({
        acc <- readRDS(fname)
    }, error = function(e) {
        logMessage('privacy:stmt:no:storage', logLInfo, paste('Privacy statement storage does not exist: "', fname, '"', sep='', collapse=''))
    }, silent=TRUE)
    acc
}

downloadText <- function(relURL) {
    serverURL <- adrServerURL()
    serverURL <- paste(serverURL, relURL, sep=if (endsWith(serverURL, '/')) '' else '/', collapse='')
    logMessage('privacy:stmt:get', logLNote, paste('Get text document from ', serverURL, sep='', collapse=''))
    r <- GET(serverURL)
    z <- content(r, "text", encoding='UTF-8')
}

adrGetPrivacyStmtTxt <- function() {
    logMessage('privacy:stmt:get', logLNote, paste('Get Privacy Statement document from OPX server', sep='', collapse=''))
    downloadText("privacy.txt")
}

adrShowPrivacyStmt <- function() {
    ps <- adrGetPrivacyStmtTxt()
    cat(ps)
}

adrGetTermsAndConditionsTxt <- function() {
    logMessage('terms:stmt:get', logLNote, paste('Get Terms and Conditions document from OPX server', sep='', collapse=''))
    downloadText("terms.txt")
}

adrShowTermsAndConditions <- function() {
    ps <- adrGetTermsAndConditionsTxt()
    cat(ps)
}

adrAcceptLegalDocs <- function() {
    fname <- adrGetAcceptLegalDocsStorage()
    acc <- TRUE
    cat(paste("OPX server Terms and Conditions of Use and Privacy Statement accepted.\n", collapse='', sep=''))
    saveRDS(acc, fname)
}

adrQueryAcceptLegalDocs <- function() {
    acc <- FALSE
    cont <- TRUE
    fname <- adrGetAcceptLegalDocsStorage()
    while (cont) {
        cat(paste("Code will be sent to the OPX server for differentiation.\n",
                  "Please accept the OPX server Terms and Conditions of Use and Privacy Statement, before proceeding.\n", collapse='', sep=''))
        r <- readline(paste("Please type:\n",
                            " (y)es          - to accept both documents,\n",
                            " (n)o           - to reject,\n",
                            " (v)iew         - to view the Terms and Conditions of Use, or\n",
                            " (p)rivacy      - to view the Privacy Statement on the console.\n",
                            "Your Answer: ", collapse='', sep=''))
        if (r == "y" || r == "yes") {
            cont <- FALSE
            acc <- TRUE
            saveRDS(acc, fname)
        } else if (r == "n" || r == "no") {
            cont <- FALSE
            acc <- FALSE
            saveRDS(acc, fname)
        } else if (r == "v" || r == "view") {
            cont <- TRUE
            adrShowTermsAndConditions()
        } else if (r == "p" || r == "privacy") {
            cont <- TRUE
            adrShowPrivacyStmt()
        } else {
            cat("Invalid reply, sorry.\n")
            cont <- TRUE
        }
    }
    acc
}

adrGetAcceptLegalDocs <- function() {
    acc <- adrLookupAcceptLegalDocs()
    if (!acc) {
        if (interactive()) {
            acc <- adrQueryAcceptLegalDocs()
        } else {
            logMessage('privacy:stmt:no:answer', logLWarn,
                       paste('Cannot run in batch mode, the terms and conditions have not been accepted'))
        }
    }
    acc
}

transformText.Headers <- function() {
    add_headers(`X-ADR-Version` = adrVersion())
}

adrServerVersion <- function() {
    serverVersion.HTTR()
}

serverVersion.HTTR <- function() {
    acc <- adrGetAcceptLegalDocs()

    if (!acc) {
        logMessage('transformText:no:consent', logLError,
                   paste('Cannot contact the OPX server when the legal documents have not been accepted'))
        return(NULL)
    }

    legalDocAcceptAnswer <- if (adrGetAcceptLegalDocs()) 'yes' else 'no'

    params <- list(`method-version` = '', output = 'txt', `accept-terms-of-use-and-privacy-stmt` = legalDocAcceptAnswer)

    serverURL <- adrTransformURL()
    logMessage('transform:httr:post', logLNote,
               paste('Post HTTR request for version info to \'', serverURL, '\', ', collapse='', sep=''))
    
    t0 <- proc.time()
    r <- ''
    tryCatch(
        {
            r <- POST(serverURL, transformText.Headers(), body = params)
            stop_for_status(r)
        },
        error = function (e) print(e)
    )
    t1 <- proc.time()
    z <- content(r, "text", encoding='UTF-8')
    logMessage('transform:httr:response', logLNote, paste('HTTR response after ', signif((t1 - t0)[['elapsed']],3), ' s, ', nchar(z), ' chars', collapse='', sep=''))
    z
}

transformText.HTTR <- function(inText, mode="r-ad-fm", params = list()) {
    acc <- adrGetAcceptLegalDocs()

    if (!acc) {
        logMessage('transformText:no:consent', logLError,
                   paste('Cannot contact the OPX server when the legal documents have not been accepted'))
        return(NULL)
    }

    legalDocAcceptAnswer <- if (adrGetAcceptLegalDocs()) 'yes' else 'no'

#    show(params)
    params <- toKeyValueParams(params)
    params <- append(params, list(mode = mode, output = 'xml', `method-tcproc` = '', `accept-terms-of-use-and-privacy-stmt` = legalDocAcceptAnswer))
#    print(paste("transformText(POST, ", paste(params, collapse='', sep=''), ")"))

    if (is.function(inText)) inText <- deparse(inText)
#    show(list(tr.t=paste(inText)))
    
    filedata <- paste(inText, collapse='\n', sep='')
    formdata <- append(params, list(data1 = filedata))
    serverURL <- adrTransformURL()
    logMessage('transform:httr:post', logLNote,
               paste('Post HTTR request for "', mode, '" to \'', serverURL, '\', ',
                     nchar(paste(inText,collapse='',sep='')), ' chars', collapse='', sep=''))
    logMessage('transform:httr:post:params', logLDebug,
               paste('Request parameters: ', paste(names(params), params, sep='=', collapse=' & '), collapse='', sep=''))
    
    t0 <- proc.time()
    r <- ''
    tryCatch(
        {
            r <- POST(serverURL, transformText.Headers(), body = formdata, encode = 'multipart')
            stop_for_status(r)
        },
        error = function (e) print(e)
    )
    t1 <- proc.time()
    z <- content(r, "text", encoding='UTF-8')
    logMessage('transform:httr:response', logLNote, paste('HTTR response after ', signif((t1 - t0)[['elapsed']],3), ' s, ', nchar(z), ' chars', collapse='', sep=''))
    z
}

adrTransform <- function(x, ...) {
    if (is.function(x)) {
        x <- getSource(x)
    }
    r <- tryCatch( 
        do.call(transformText.HTTR, list(x, ...)),
        error = function (e) print(e)
    )
    if (is.null(r)) {
        logMessage('transformText:fail', logLError, paste('Failed to transform text'))
    }
    r
}

diffText <- function(inTxt, mode = 'fm', params = list()) {
    if (is.expression(inTxt) || is.function(inTxt)) {
        inTxt <- getSource(inTxt)
    }
    params['tmp-prefix'] <- paste(if (mode == 'fm') 'f.' else 'r.', sep='', collapse='')
    mode <- paste(c("r-ad-", mode), collapse="", sep="")
    logMessage('diffText:source', logLChat,
               paste('diffText: mode: ', mode,
                     ', params: ', paste(names(params), params, sep='=', collapse=','),
                     ', code:\n', paste0(inTxt, collapse='\n'),
                     sep='', collapse=''))
    r <- adrTransform(inTxt, mode, params)
    logMessage('diffText:result', logLChat, paste('diffText ', mode, ' code:\n', r, sep='', collapse=''))
    errreg <- '[0-9]+ errors and [0-9]+ warnings occured during'
    errs <- grep(errreg, r)
    if (length(errs)) {
        splt <- strsplit(r, '\n')[[1]]
        logMessage('diffText:failed', logLError,
                   cat(sprintf('Transformation failed: %s\n', splt[grepl(errreg, splt)])))
        ## cat(paste(grep('error', splt, value = TRUE), collapse='\n', sep=''))
        cat(r)
    }
#    print("Original code: ")
#    print(paste(inTxt, sep=''), quote=FALSE)
#    print("Diff. code: ")
#    writeLines(r)
    r
}

adrDiffText <- function(inTxt, params = list()) {
    diffText(inTxt, 'fm', params)
}
adrAdjText <- function(inTxt, params = list()) {
    diffText(inTxt, 'rm', params)
}

adrR2XML <- function(txt) {
    if (is.function(txt)) {
        txt <- getSource(txt)
    }
    adrTransform(txt, 'r-lang-xml')
}

hashEnv <- function( keys ) {
    result <- new.env( hash = TRUE, parent = emptyenv(), size = length( keys ) )
    for( key in keys ) {
        result[[ key ]] <- NA
    }
    return( result )
}

getFuncEnvUUID <- function(handle) {
    e <- environment(handle)
    uuid <- e[['.adr']]
    if (is.null(uuid)) {
        e[['.adr']] <- UUIDgenerate()
        uuid <- e[['.adr']]
#        print(paste('Provided environment with ADR ID:', uuid))
#        show(e)
    }
    uuid
}

getFunctionNameFromHandle <- function(h, fname) {
#    if (is.function(h)) deparse(h)
#    else
        fname
}
getFunctionKey <- function(fhandle) {
    fkey <- paste(getSource(fhandle), sep="", collapse="")
    fkey
}
getFunctionKeyMode <- function(fhandle, mode) {
    fkey <- paste(getFunctionKey(fhandle), '--m', mode, sep="", collapse="")
    fkey
}
getParamsCacheKey <- function(fhandle, mode, actlist, params, fname) {
    np <- sort(names(params))
    plist <- paste(np, params[np], sep='=', collapse=';')
#    alist <- paste(actlist, sep='__', collapse='')
    paste(mode, plist, '', sep='', collapse='')
}
getActlistCacheKey <- function(actlist) {
    alist <- paste(actlist, sep='__', collapse='')
    alist
}

mkHashEnv <- function() {
    local({
        karr <- list()
        function(key, mode='hash') {
            if (mode == 'hash') {
                ent <- karr[[key]]
                if (is.null(ent)) {
                    n <- length(karr)+1
                    karr[[key]] <<- n
                    ent <- karr[[key]]
                }
                ent
            } else if (mode == 'dump') {
                karr
            } else if (mode == 'name') {
                names(karr)[[as.integer(key)]]
            }
        }
    })
}

getTransformCacheKeyP <- function(fhandle, mode, actlist, params, fname) {
    paste(getFunctionKey(fhandle),
          getParamsCacheKey(fhandle, mode, actlist, params, fname), sep='', collapse='')
}

getTransformCacheKeyL <- function(fhandle, mode, actlist, params, fname) {
    paste(getTransformCacheKeyP(fhandle, mode, actlist, params, fname),
          getActlistCacheKey(actlist), sep='', collapse='')
}

getTransformCacheKey <- local({
    shortList <- mkHashEnv()
    function(fhandle, mode, actlist, params, fname) {
        as.character(shortList(getTransformCacheKeyL(fhandle, mode, actlist, params, fname)))
    }
})

envAncList <- function(e) {
    if (!is.environment(e)) {
        list()
    } else if (identical(e, .GlobalEnv)) {
        list(e)
    } else {
        c(list(e), envAncList(parent.env(e)))
    }
}

printEnvNames <- function(env = NULL, n = 1) {
    if (is.null(env)) {
        env <- parent.frame()
    }
    print(paste('environment level ', n, environmentName(env), ', names: ', paste(names(env),sep='',collapse=', ')))
    if (environmentName(env) != "R_GlobalEnv" && environmentName(env) != "R_EmptyEnv")
        printEnvNames(parent.env(env), n + 1)
}

dsource <- function(scode, actlist=c(), mode = 'fm', params = list(), local = parent.frame()) {
    if (length(actlist) > 0) {
        params$independents = paste(actlist, collapse=',', sep='')
    }
#    show(params)
    res <- diffText(scode, mode, params)
    expr <- NULL
    func <- NULL
    if (!is.null(attr(res, "status")) && attr(res, "status") != 0)
        stop('diffText produced a non-zero exit: (', attr(res, "errmsg"), '', res)
    if (!is.null(res)) {
        sf <- srcfile("res")
        tryCatch(
            expr <- parse(text = res, keep.source=TRUE, srcfile = sf),
            error = function(e) {
                expr <<- NULL
            })
    }
    if (is.null(expr)) {
        logMessage('dsource:result:fail', logLError, paste('Cannot parse the result code'))
        if (!is.null(res)) {
            lines <- strsplit(res, '\n')[[1]]
            dp <- getParseData(sf)
            error <- which(dp$token == "'error'")
            n <- nrow(dp)
            ll <- dp[n,'line2']
            bll <- min(which(dp$line2 == ll))
            oklines <- lines[1:ll]
            logMessage('dsource:result:parse:text', logLError, paste('\n', paste(oklines, sep='', collapse='\n'), sep='', collapse=''))
            if (!is.null(dp[n,'col2'])) {
                nspace <- max(0, dp[n,'col2']-2)
                logMessage('dsource:result:parse:error', logLError, 
                           paste('error:line:', dp[n,'line2'], ':', dp[n,'col2'], ':\n',
                                 paste(paste(rep(' ', nspace),sep='', collapse=''), "'", dp[n,'text'], "'", sep='', collapse=''),
                                 sep='', collapse=''))
            }
        }
    } else {
        logMessage('dsource:result', logLInfo, paste('', getSource(expr)))
        func <- eval(expr, envir = local)
#        show(func)
#        show(as.list(func))
#        show(attributes(func))
        attr(func, 'srcref') <- NULL
    }
    func
}

actParamsStr <- function(handle, actlist) {
    if (length(actlist)==0) "all parameters"
    else {
        fh <- formals(handle)
        if (is.null(fh) || '...' %in% names(fh)) {
            paste(actlist, sep='', collapse=', ')
        } else {
            paste(names(fh)[actlist], sep='', collapse=', ')
        }
    }
 }

diffFunction <- local({
    diffFuncDict <- hashEnv(list())
    diffFuncPDict <- hashEnv(list())
    function(handle, actlist, fname, mode = 'fm', params = list(), envir = parent.frame()) {

        if (mode == 'cache') {
            return(list(diffFuncDict, diffFuncPDict))
        } else if (mode == 'clear-cache') {
            diffFuncDict <<- hashEnv(list())
            diffFuncPDict <<- hashEnv(list())
            return(0)
        } else if (mode == 'set-cache') {
            diffFuncDict <<- handle[[1]]
            diffFuncPDict <<- handle[[2]]
            return(0)
        }

        if (is.null(handle)) {
            stop('handle must not be NULL')
        }

        ## if (file.exists('.adr-cache.Rda')) {
        ##     load('.adr-cache.Rda')
        ##     diffFuncDict <<- savelist[[1]]
        ##     diffFuncPDict <<- savelist[[2]]
        ## }
        
        env <- if (is.null(envir)) environment(handle) else envir

        skey <- getTransformCacheKey(handle, mode, actlist, params, fname)
        fskey <- getFunctionKey(handle)
        pskey <- getTransformCacheKeyP(handle, mode, actlist, params, fname)
        vskey <- getParamsCacheKey(handle, mode, actlist, params, fname)
        actlstr = actParamsStr(handle, actlist)
        entry <- diffFuncDict[[pskey]]
        if (is.null(entry)) {
            entry <- diffFuncPDict[[skey]]
        } else {
            if (is.expression(entry)) {
                entry <- eval(entry)
            }
        }
        if (is.null(entry)) {
            logMessage('diffFunction:cache:miss', logLInfo, paste('Cache: ', mode, " code for '", paste0(fname, collapse=' '), "', w.r.t. ", actlstr, " not found, key: ", skey, sep='', collapse=''))
            logMessage('diffFunction:cache:miss:fkey', logLDebug, paste('Cache function key: ', fskey, sep='', collapse=''))
            logMessage('diffFunction:cache:miss:key', logLDebug, paste('Cache key: ', vskey, sep='', collapse=''))
            s <- getSource(handle)
#            show(params)
            entry <- dsource(s, actlist, mode, params, local = env)
            diffFuncPDict[[skey]] <- entry
        } else {
            logMessage('diffFunction:cache:hit', logLDebug, paste('Cache: ', mode, " code for '", paste0(fname, collapse=' '), "', w.r.t. ", actlstr, ", key: ", skey, ", already present in the dictionary", sep='', collapse=''))
            logMessage('diffFunction:cache:hit:fkey', logLDebug, paste('Cache function key: ', fskey, sep='', collapse=''))
            logMessage('diffFunction:cache:hit:key', logLDebug, paste('Cache key: ', vskey, sep='', collapse=''))
        }
        if (is.null(entry)) {
            logMessage('diffFunction:cache:hit', logLError, paste("Failed to obtain ", mode, " code for '", fname, "', w.r.t. ", actlstr, " ", sep='', collapse=''))
        } else {
            environment(entry) <- env
        }
#        savelist = list(diffFuncDict, diffFuncPDict)
#        save(savelist, file = '.adr-cache.Rda')
#        show(as.list(savelist[[1]]))
#        show(as.list(savelist[[2]]))
        entry
    }
})

d <- function(handle, actlist=c(), fname = getFunctionNameFromHandle(handle, substitute(handle)),
              envir = parent.frame()) {
    diffFunction(handle, actlist, fname, envir = envir)
}

adj <- function(handle, actlist=c(), fname = getFunctionNameFromHandle(handle, substitute(handle)),
                envir = parent.frame(), adjmode = 'adj') {
    params <- list(`adjoint-mode` = adjmode)
    diffFunction(handle, actlist, fname, mode = 'rm', params = params, envir = envir)
}

adj.rec <- function(..., fname = getFunctionNameFromHandle(handle, substitute(handle)), envir = parent.frame()) {
    adj(..., fname = fname, envir = envir, adjmode = 'rec')
}

adj.ret <- function(..., fname = getFunctionNameFromHandle(handle, substitute(handle)), envir = parent.frame()) {
    adj(..., fname = fname, envir = envir, adjmode = 'ret')
}

adrCache <- function(x = NULL, mode = 'cache') {
    diffFunction(x, mode = mode)
}

adrOptionDefs <- function() {
    list(independents = adrGetOption('independents'),
         fdStep = adrGetOption('fdStep'),
         admode = adrGetOption('admode'),
         vectormode = adrGetOption('vectormode'),
         `vectormode-switchover` = adrGetOption('vectormode-switchover', 10),
         modeFlags = adrGetOption('modeFlags', 0),
         trace = adrGetOption('trace', 0),
         functionResult = NULL,
         maxOrder = 1)
}

mergeLists <- function (l1, l2) {
    r <- do.call(list, l1)
    for (k in names(l2)) {
        if (!(k %in% names(r))) {
            r[k] <- list(l2[[k]])
        }
    }
    r
}

adrOptions <- function(...) {
    inargs <- list(...)
    for (k in names(inargs)) {
        if (k == 'i') {
            inargs$independents <- inargs$i
            inargs$i <- NULL
        } else if (k == 'h') {
            inargs$fdStep <- inargs$h
            inargs$h <- NULL
        }
    }
    mergeLists(inargs, adrOptionDefs())
}

adrDiffArgs <- function(args, seed=1) {
    grads <- do.call(createSeededGradientsFor, append(list(seed), args))
    if (!is.null(names(args))) {
        names(grads) <- names(args)
    }
    grads
}

adr.is.numeric <- function(x) is.numeric(x) || is.complex(x)

maxNTraceParams <- 2^0

setTraceLevels <- function(options) {
    ll <- adrGetOption('loglevels', list())
    trval <- adrGetOption('trace', if (is.null(options$trace)) 0 else options$trace)
    if (trval >= maxNTraceParams) {
        if (bitwAnd(trval, 1)) {
            adrSetLoglev('diffFunction:cache:miss' = 15, 
                         'transform:httr:post' =  15)
        }
        if (bitwAnd(trval, 2)) {
            adrSetLoglev('diffText:source' =  15,
                         'diffText:result' =  15)
        }
        if (bitwAnd(trval, 4)) {
        }
        if (bitwAnd(trval, 8)) {
            adrSetLoglev('transform:httr:post:params' =  15)
        }
        if (bitwAnd(trval, 16)) {
            adrSetLoglev('diffFunction:cache:hit' = 15,
                         'agetfun:adef' = 15)
            adrSetLoglev('dgetfun:ddef:found' = 15)
        }
        if (bitwAnd(trval, 32)) {
            adrSetLoglev('diffFunction:cache:miss:key' = 15)
            adrSetLoglev('diffFunction:cache:hit:key' = 15)
            adrSetLoglev('dgetfun:ddef:not:found' = 15)
        }
        if (bitwAnd(trval, 64)) {
        }
    }
    ll
}

adrDiffFor <- function(func, arguments, seed = 1, options = adrOptions()) {
    fname <- substitute(func)
    oldLL <- setTraceLevels(options)
    if (length(options$independents) == 0) {
        options$independents <- 1 # fixme here all now
    }
    options$independents <- options$independents[options$independents <= length(arguments)]
    if (!all(sapply(arguments[options$independents], is.numeric) | sapply(arguments[options$independents], is.complex))) {
        stop('all indep. arguments must be numeric')
    }
    if (prod(size(seed)) == 1) {
        seed <- seed * eye(do.call(totalNumel, arguments[options$independents]))
    }
    srcenv <- environment(func)
    globalADROptions('srcenv', srcenv)
    ndd = size(seed)[[2]]
    globalNDD(ndd)
    if (is.null(options$vectormode)) {
       if (ndd >= options$`vectormode-switchover`) {
          globalADMode(1)
       } else {
          globalADMode(0)
       }
    } else if (options$vectormode) {
       globalADMode(1)
    } else {
       globalADMode(0)
    }
    if (globalADMode() == 1) {
        grads <- adrDiffArgs(arguments[options$independents], seed)
        adr <- dcall(func, grads, arguments, options$independents, fname = fname)
        if (is.null(adr)) {
            logMessage('adrDiffFor:dcall:null', logLError, paste0("Could not evaluate the derivative code"))
            return(0)
        }
#        show('vector mode result:')
#        show(adr)
        J <- jacFor(adr[[1]])
        if (class(adr[[1]]) != 'advec') {
            logMessage('adrDiffFor:result:not:advec', logLWarn,
                       paste0('Derivative result has unexpected class ', class(adr[[1]]),
                              ', this is not `advec`'))
        }
        if (!adr.is.numeric(adr[[2]])) {
            logMessage('adrDiffFor:result:not:numeric', logLWarn, 
                       paste0('Function result has unexpected class ', class(adr[[2]]),
                              ', this is not numeric'))
        }
        logMessage('adrDiffFor:result', logLDebug, paste0(class(adr), ', ', class(adr[[1]]), ', ', class(adr[[2]])))
        logMessage('adrDiffFor:result', logLDebug, paste0('class ', paste0(class(adr)),
                                                             ' jsize ', paste(size(jacFor(adr[[1]])), collapse=', '),
                                                             ' dsize ', paste(size(adr[[1]]), collapse=', '),
                                                             ' size ', paste(size(adr[[2]]), collapse=', ')))
    } else {
        for (k in 1:ndd) {
          seedv <- if (ndd > 1) seed[,k] else seed
          grads <- adrDiffArgs(arguments[options$independents], seedv)
          adr <- dcall(func, grads, arguments, options$independents, fname = fname)
          if (k == 1) {
              if (!adr.is.numeric(adr[[1]])) {
                  logMessage('adrDiffFor:deriv:not:numeric', logLWarn,
                             paste0('Derivative result has unexpected class ', class(adr[[1]]),
                                    ', this is not numeric'))
              }
              if (!adr.is.numeric(adr[[2]])) {
                  logMessage('adrDiffFor:result:not:numeric', logLWarn,
                             paste0('Function result has unexpected class ', class(adr[[2]]),
                                    ', this is not numeric'))
              }
              logMessage('adrDiffFor:result', logLDebug, paste0(class(adr), ', ', class(adr[[1]]), ', ', class(adr[[2]])))
              logMessage('adrDiffFor:result', logLDebug, paste0('class ', paste0(class(adr)),
                                                                   ' jsize ', paste(size(jacFor(adr[[1]])), collapse=', '),
                                                                   ' dsize ', paste(size(adr[[1]]), collapse=', '),
                                                                   ' size ', paste(size(adr[[2]]), collapse=', ')))
             J <- matrix(0, ncol = size(seed)[[2]], nrow = totalNumel(adr[[2]]))
          }
          if (is.null(adr)) {
              logMessage('adrDiffFor:dcall:null', logLError, paste0("Could not evaluate the derivative code"))
              return(0)
          }
#          show(adr)
          J[,k] <- jacFor(adr[[1]])
        }
    }
    do.call(adrSetLoglev, oldLL)
    list(J=J, f=adr[[2]])
}

adrDiffRev <- function(func, arguments, seed = 1, options = adrOptions(), fname = substitute(func)) {
    oldLL <- setTraceLevels(options)
    if (length(options$independents) == 0) {
        options$independents <- 1 # fixme here all now
    }
    options$independents <- options$independents[options$independents <= length(arguments)]
    ndargs <- length(options$independents)
    if (is.null(options$functionResult)) {
        options$functionResult <- do.call(func, arguments)
    }
    if (is.null(options$functionResult) || prod(safedim(options$functionResult)) == 0) {
        stop('adr:functionResult_is_null_or_empty');
    }
    if (!all(sapply(arguments[options$independents], is.numeric) | sapply(arguments[options$independents], is.complex))) {
        stop('all indep. arguments must be numeric')
    }
    if (!(is.numeric(options$functionResult) || is.complex(options$functionResult))) {
        stop('function result must be numeric')
    }
    if (prod(size(seed)) == 1) {
        seed <- seed * eye(do.call(totalNumel, list(options$functionResult)))
    }
    srcenv <- environment(func)
    globalADROptions('srcenv', srcenv)
    ndd = size(seed)[[1]]
#    show(paste('ndd = ', ndd))
    globalNDD(ndd)
    adrSetOption('a_zeros-mode', 0)
    if (is.null(options$vectormode)) {
       if (ndd >= options$`vectormode-switchover`) {
          globalADMode(1)
       } else {
          globalADMode(0)
       }
    } else if (options$vectormode) {
       globalADMode(1)
    } else {
       globalADMode(0)
    }
    if (globalADMode() == 1) {
#        show('Vector mode')
        grads <- adrDiffArgs(list(options$functionResult), t(seed))[[1]]
        adr <- acall(func, grads, arguments, options$independents, fname = fname)
        if (is.null(adr)) {
            logMessage('adrDiffRev:dcall:null', logLError, paste0("Could not evaluate the derivative code"))
            res <- 0
            attr(res, 'error') <- TRUE
            return(res)
        }
        J <- t(do.call(jacFor, adr[[1]]))
        if (class(adr[[1]][[1]]) != 'advec') {
            logMessage('adrDiffRev:result:not:advec', logLWarn,
                       paste0('Derivative result has unexpected class ', class(adr[[1]]),
                              ', this is not `advec`'))
        }
        if (!adr.is.numeric(adr[[2]])) {
            logMessage('adrDiffRev:result:not:numeric', logLWarn, 
                       paste0('Function result has unexpected class ', class(adr[[2]]),
                              ', this is not numeric'))
        }
        logMessage('adrDiffRev:result', logLDebug, paste0(class(adr), ', ', class(adr[[1]]), ', ', class(adr[[2]])))
        logMessage('adrDiffRev:result', logLDebug, paste0('class ', paste0(class(adr)),
                                                             ' jsize ', paste(size(jacFor(adr[[1]])), collapse=', '),
                                                             ' dsize ', paste(size(adr[[1]]), collapse=', '),
                                                             ' size ', paste(size(adr[[2]]), collapse=', ')))
    } else {
        for (k in 1:ndd) {
          seedv <- if (ndd > 1) seed[k,] else seed
          grads <- adrDiffArgs(list(options$functionResult), t(t(c(seedv))))[[1]]
          adr <- acall(func, grads, arguments, options$independents, fname = fname)
          if (k == 1) {
              if (!adr.is.numeric(adr[[1]][[1]])) {
                  logMessage('adrDiffFor:deriv:not:numeric', logLWarn,
                             paste0('Derivative result has unexpected class ', class(adr[[1]]),
                                    ', this is not numeric'))
              }
              if (!adr.is.numeric(adr[[2]])) {
                  logMessage('adrDiffFor:result:not:numeric', logLWarn,
                             paste0('Function result has unexpected class ', class(adr[[2]]),
                                    ', this is not numeric'))
              }
              logMessage('adrDiffFor:result', logLDebug, paste0(class(adr), ', ', class(adr[[1]]), ', ', class(adr[[2]])))
              logMessage('adrDiffFor:result', logLDebug, paste0('class ', paste0(class(adr)),
                                                                   ' jsize ', paste(size(jacFor(adr[[1]])), collapse=', '),
                                                                   ' dsize ', paste(size(adr[[1]]), collapse=', '),
                                                                   ' size ', paste(size(adr[[2]]), collapse=', ')))
             J <- matrix(0, ncol = do.call(totalNumel, arguments[options$independents]), nrow = size(seed)[[1]])
          }
          if (is.null(adr)) {
              logMessage('adrDiffFor:dcall:null', logLError, paste0("Could not evaluate the derivative code"))
              res <- 0
              attr(res, 'error') <- TRUE
              return(res)
          }
          Jstripe <- t(do.call(jacFor, adr[[1]][1:length(options$independents)]))
          J[k,] <- Jstripe
        }
    }
    do.call(adrSetLoglev, oldLL)
    list(J=J, f=adr[[2]])
}

l2apply_l <- function(op, a, b) {
    if (is.list(a)) {
        lapply(1:length(a), function(i) { l2apply(op, a[[i]], b) })
    } else {
        op(a, b)
    }
}
l2apply <- function(op, a, b) {
    if (is.list(a)) {
        lapply(1:length(a), function(i) { l2apply(op, a[[i]], b[[i]]) })
    } else {
        op(a, b)
    }
}
lsum <- function(a, b) {
    l2apply(`+`, a, b)
}
lflat <- function(a) {
    if (is.list(a)) {
            do.call(c, lapply(a, lflat))
    } else {
        a
    }
}

evalFD <- function(fcn, arguments, dir, coeffs, options = adrOptions()) {
    nsteps <- length(coeffs)
    centi <- floor(nsteps / 2)+1
    rescent <- do.call(fcn, arguments)

#    show(list(coeffs = coeffs, centi = centi, ccoeff = coeffs[[centi]]))
    res <- lflat(rescent) * coeffs[[centi]]

    ch <- options$fdStep^(1/options$maxOrder)
    
    for (i in 1:length(coeffs)) {
        coeff <- coeffs[[i]]

        step = i - centi
        if (step == 0) next

        delta = step * ch
        
#        show(list(i=0, step = step, delta = delta, dir = dir, coeff = coeff))

        offs <- 1
        argsci <- arguments
        for (k in options$independents) {
            nk <- numel(arguments[[k]])
            argsci[[k]] <- arguments[[k]] + delta * dir[offs:(offs+nk-1)]
            offs <- offs + nk
        }

#        show(list(da = arguments[[1]] - argsci[[1]]))
        resci <- do.call(fcn, argsci)

#        show(list(resci = resci))
        
        res <- res + coeff * lflat(resci)
    }

#    show(list(res = res))
    
    res <- res / (ch^options$maxOrder)
#    show(list(res = res))
    res
}


adrDiffFD <- function(fcn, arguments, seed = 1, options = adrOptions()) {
    if (length(options$independents) == 0) {
        options$independents <- 1 # fixme here all now
    }
    options$independents <- options$independents[options$independents <= length(arguments)]
    nactins <- do.call(totalNumel, arguments[options$independents])
    if (prod(dim(seed)) == 1) {
        seed <- seed * sparseMatrix(i=1:nactins, j=1:nactins, x=1, dims=c(nactins,nactins))
    }
    ns1 <- size(seed)[1]
    if (nactins != ns1) {
        stop(paste('The seed matrix must have', nactins, 'rows but has', ns1))
    }
    ndd <- size(seed)[2]
    res <- do.call(fcn, arguments)
    nres <- numel(lflat(res))
    if (!all(sapply(arguments[options$independents], is.numeric) | sapply(arguments[options$independents], is.complex))) {
        stop('all indep. arguments must be numeric')
    }
                                        #    show(list(func.res = res, t = class(res), nout = nres))

    coeffList = list(
        order1 = c(-0.5, 0, 0.5),
        order2 = c(1,   -2,   1)
    )

    coeffs <- coeffList[[options$maxOrder]]
    
    J <- array(0, c(nres, ndd))
    for (i in 1:ndd) {
        seedCol = seed[,i];
        offs <- 1
        
        Jcol <- evalFD(fcn, arguments, dir = seedCol, coeffs = coeffs, options = options)
        
        J[,i] <- Jcol
    }

    list(J=J, f=res)
}

adrHessFDSym <- function(func, arguments, Y = 1, U = 1, V = 1, options = adrOptions(i = 1)) {
    J <- adrDiffFD(func, arguments, V, options)

    nin <- do.call(totalNumel, arguments[options$independents])
    nout <- do.call(totalNumel, append(list(), J$f))
    sarr <- array(0, c(nin, (nin * (nin+1))/2))

    offs <- 1
    for (i in 1:nin) {
        sarr[i, offs] <- 1
        offs <- offs + 1
    }
    if (nin > 1) {
        for (i in 1:(nin-1)) {
            for (j in (i+1):nin) {
                sarr[i, offs] <- 1
                sarr[j, offs] <- 1
                offs <- offs + 1
            }
        }
    }

    options$maxOrder = 2
    T2 <-  adrDiffFD(func, arguments, seed = sarr, options = options)
    Hess <- array(0, c(nout, nin, nin))

    if (nin > 1) {
        offs <- nin+1
        for (i in 1:(nin-1)) {
            for (j in (i+1):nin) {
                Hess[TRUE,i,j] = (T2$J[TRUE,offs] - T2$J[TRUE,i] - T2$J[TRUE,j]) / 2
                offs <- offs + 1
            }
        }
    }
    Hess <- Hess + aperm(Hess, c(1, 3, 2))
    for (i in 1:nin) {
        Hess[TRUE,i,i] = T2$J[TRUE, i]
    }

    for (k in 1:nout) {
        Hess[k,,] <- U %*% Hess[k,,] %*% V
    }
    
    append(list(H = drop(Hess)), J)
}

adrHessFDV <- function(func, arguments, Y = 1, U = 1, V = 1, options = adrOptions(i = 1)) {
    J <- adrDiffFD(func, arguments, V, options)

    nin <- do.call(totalNumel, arguments[options$independents])
    nout <- do.call(totalNumel, append(list(), J$f))
    
    nv <- nrow(U) * ncol(V) + nrow(U) + ncol(V)

    sarr <- array(0, c(nin, nv))

    offs <- 1
    for (i in 1:nrow(U)) {
        sarr[, offs] <- U[i,]
        offs <- offs + 1
    }
    for (i in 1:ncol(V)) {
        sarr[, offs] <- V[,i]
        offs <- offs + 1
    }
    for (i in 1:nrow(U)) {
        for (j in 1:ncol(V)) {
            sarr[, offs] <- V[,j] + U[i,]
            offs <- offs + 1
        }
    }

    options$maxOrder = 2
    T2 <-  adrDiffFD(func, arguments, seed = sarr, options = options)

    Hess <- array(0, c(nrow(U), ncol(V), nout))

    offs <- nrow(U) + ncol(V) + 1
    for (i in 1:nrow(U)) {
        for (j in 1:ncol(V)) {
            Hess[i,j,TRUE] = (T2$J[TRUE,offs] - T2$J[TRUE,i] - T2$J[TRUE,j+nrow(U)]) / 2
            offs <- offs + 1
        }
    }

    append(list(H = drop(Hess)), J)
}

adrHessFD <- function(func, arguments, Y = 1, U = 1, V = 1, options = adrOptions(i = 1)) {
    if (length(options$independents) == 0) {
        options$independents
    }
    options$independents <- options$independents[options$independents <= length(arguments)]
    nin <- do.call(totalNumel, arguments[options$independents])

    nsym <- nin * (nin + 1) / 2
    if (identical(U, 1)) {
        U <- diag(nin)
    }
    if (identical(V, 1)) {
        V <- diag(nin)
    }
    if (ncol(U) != nin) {
        stop(paste('Hessian left seed U must have as many columns as there are inputs, but has', ncol(U), '!=', nin))
    }
    if (nrow(V) != nin) {
        stop(paste('Hessian right seed V must have as many rows as there are inputs, but has', nrow(V), '!=', nin))
    }
    nv <- nrow(U) * ncol(V) + nrow(U) + ncol(V)

    if (nsym < nv) {
        H <- adrHessFDSym(func, arguments, Y, U, V, options)
    } else {
        H <- adrHessFDV(func, arguments, Y, U, V, options)
    }
    nout <- do.call(totalNumel, append(list(), H$f))
    if (!identical(Y, 1)) {
        if (identical(Y, 1)) {
            Y <- diag(nout)
        } else {
            ndd  <- nrow(Y)
            n2  <- ncol(Y)
            if (is.na(n2) || n2 != nout) {
                stop(paste('Hessian adjoint seed Y must have as many columns as there are outputs, but has', ncol(Y), '!=', nout))
            }
        }
        H$H <- drop(adr_reshape(Y %*% adr_reshape(H$H, c(nout, nrow(U) * ncol(V))), c(nrow(Y), nrow(U), ncol(V))))
    }
    H
}


adrTaylorFor <- function(func, arguments, seed = 1, options = adrOptions()) {
    fname <- substitute(func)
    oldLL <- setTraceLevels(options)
    if (length(options$independents) == 0) {
        options$independents <- 1 # fixme here all now
    }
    options$independents <- options$independents[options$independents <= length(arguments)]
    if (!all(sapply(arguments[options$independents], is.numeric) | sapply(arguments[options$independents], is.complex))) {
        stop('all indep. arguments must be numeric')
    }
    if (prod(size(seed)) == 1) {
        seed <- seed * eye(do.call(totalNumel, arguments[options$independents]))
    }
    srcenv <- environment(func)
    globalADROptions('srcenv', srcenv)
    ndd = size(seed)[[2]]
    globalNDD(ndd)
    ## if (is.null(options$vectormode)) {
    ##    if (ndd >= options$`vectormode-switchover`) {
    ##       globalADMode(1)
    ##    } else {
    ##       globalADMode(0)
    ##    }
    ## } else if (options$vectormode) {
    ##    globalADMode(1)
    ## } else {
    ##    globalADMode(0)
    ## }
    globalADMode(1)
    adrGlobalOptions(taylormode = 1)
    if (globalADMode() == 1) {
        grads <- adrDiffArgs(arguments[options$independents], seed)

        newargs <- lapply(1:length(arguments), function(i) {
            if (i %in% options$independents) {
                adtay(arguments[[i]], ndd, 1)
            } else {
                arguments[[i]]
            }
        })
        lapply(1:length(options$independents), function(i) {
            newargs[[options$independents[[i]]]]@der[[1]] <<- grads[[i]]
        })
        names(newargs) <- names(arguments)

        adr <- do.call(func, newargs)
        
        if (is.null(adr)) {
            logMessage('adrTaylorFor:dcall:null', logLError, paste0("Could not evaluate the derivative code, result is null"))
            return(list(0,0))
        }

        if (class(adr) != 'adtay') {
            logMessage('adrDiffFor:result:not:adtay', logLWarn,
                       paste0('Derivative result has unexpected class ', class(adr),
                              ', this is not `adtay`'))
            return(list(0,0))
        }
        
#        show('vector mode result:')
#        show(adr)
        J <- jacFor(adr@der[[1]])

        logMessage('adrDiffFor:result', logLDebug, paste0(class(adr), ', ', class(adr[[1]]), ', ', class(adr[[2]])))
        logMessage('adrDiffFor:result', logLDebug, paste0('class ', paste0(class(adr)),
                                                             ' jsize ', paste(size(jacFor(adr[[1]])), collapse=', '),
                                                             ' dsize ', paste(size(adr[[1]]), collapse=', '),
                                                             ' size ', paste(size(adr[[2]]), collapse=', ')))
    } else {
        for (k in 1:ndd) {
          seedv <- if (ndd > 1) seed[,k] else seed
          grads <- adrDiffArgs(arguments[options$independents], seedv)
          if (is.null(adr)) {
              logMessage('adrDiffFor:dcall:null', logLError, paste0("Could not evaluate the derivative code"))
              return(0)
          }
#          show(adr)
          J[,k] <- jacFor(adr[[1]])
        }
    }

    do.call(adrSetLoglev, oldLL)
    list(J=J, f=adr@val)
}


adrHessRev <- function(func, arguments, Y = 1, U = 1, V = 1, options = adrOptions(i = 1)) {
    oldLL <- setTraceLevels(options)
    if (length(options$independents) == 0) {
        options$independents
    }
    options$independents <- options$independents[options$independents <= length(arguments)]
    nin <- do.call(totalNumel, arguments[options$independents])
    if (is.null(options$functionResult)) {
        options$functionResult <- do.call(func, arguments)
    }
    if (is.null(options$functionResult) || prod(safedim(options$functionResult)) == 0) {
        stop('adr:functionResult_is_null_or_empty');
    }
    if (!all(sapply(arguments[options$independents], is.numeric) | sapply(arguments[options$independents], is.complex))) {
        stop('all indep. arguments must be numeric')
    }
    if (!(is.numeric(options$functionResult) || is.complex(options$functionResult))) {
        stop(sprintf('function result must be numeric, but is class \'%s\'', class(options$functionResult)))
    }
    nout <- totalNumel(options$functionResult)
    
    if (identical(U, 1)) {
        U <- diag(nin)
    }
    if (identical(V, 1)) {
        V <- diag(nin)
    }
    if (identical(Y, 1)) {
        Y <- diag(nout)
    }
    if (ncol(U) != nin) {
        stop(paste('Hessian left seed U must have as many columns as there are inputs, but has', ncol(U), '!=', nin))
    }
    if (nrow(V) != nin) {
        stop(paste('Hessian right seed V must have as many rows as there are inputs, but has', nrow(V), '!=', nin))
    }
    nY2  <- ncol(Y)
    if (is.na(nY2) || nY2 != nout) {
        stop(paste('Hessian adjoint seed Y must have as many columns as there are outputs, but has', ncol(Y), '!=', nout))
    }


    srcenv <- environment(func)
    globalADROptions('srcenv', srcenv)

    ndd <- nout

    H <- array(0, c(nrow(Y), nin, ncol(V)))
#    show(list(Hdim = dim(H)))
    Jv <- array(0, c(nrow(Y), nin))
    J <- array(0, c(nout, ncol(V)))
    f <- array(0, nout)

    vectorMode <- FALSE
    if (is.null(options$vectormode)) {
        vectorMode <- TRUE
    } else if (options$vectormode) {
        vectorMode <- TRUE
    }

    if (vectorMode) {
        adf <- adj(func, options$independents, envir = srcenv)
        
        adrSetOption('a_zeros-mode', 12)
        adrSetOption('taylor-order', 1)
        globalADMode(1)

        dargs <- do.call(createSeededGradientsFor, append(list(V), arguments[options$independents]))

        for (adji in 1:nrow(Y)) {

            targs <- arguments
            targs[options$independents] <- lapply(1:length(options$independents), function(ai) {
                i <- options$independents[[ai]]
                x <- targs[[i]]
                r <- a_zeros(x)
#                show(list(mktargs.x=x,r=r))
                r@val <- x
                r@der[[1]] <- dargs[[ai]]
                r
            })
                                        #        a_d_z <- a_zeros(options$functionResult)
#            a_d_z <- adr_reshape(as.array(Y[adji,]), safedim(options$functionResult))
            a_z <- a_zeros(options$functionResult)
            Yrow <- as.array(Y[adji,])
            a_z@val <- adr_reshape(Yrow, safedim(options$functionResult))
                                        #        dim(a_d_z@data) <- c(numel(options$functionResult), a_d_z@ndd)
                                        #        a_d_z@data[,1] <- adr_reshape(as.array(Y[adji,]), safedim(options$functionResult))
                                        #        dim(a_d_z@data) <- c(a_d_z@sz, a_d_z@ndd)

#            show(list(adf.targs=targs, a_z = a_z))
            ares <- adf(a_z, targs, options$independents)
            
#            show(list(ares=ares))
            
            Hplane <- do.call(jacFor, lapply(ares$adj, function(t) t@der[[1]]))
                                        #        show(list(adji = adji, Hplane = Hplane))
            H[adji,,] <- Hplane
            
            if (adji == 1) {
                J <- jacFor(ares$f@der[[1]])
                f <- ares$f@val
            }
            Jv[adji,] <- do.call(jacFor, lapply(ares$adj, function(t) t@val))
        }

    } else {
        df <- d(func, options$independents)

        adf <- adj(df, 1:2, envir = srcenv)

                                        #    show(adf)
        globalADMode(0)
        adrSetOption('a_zeros-mode', 0)

        for (idir in 1:ncol(V)) {
#            cat('|')
            dargs <- do.call(createSeededGradientsFor, append(list(V[,idir]), arguments[options$independents]))

            for (adji in 1:nrow(Y)) {
                                        #        a_d_z <- a_zeros(options$functionResult)
                a_d_z <- if (ncol(Y) == 1) Y[adji,] else adr_reshape(Y[adji,], safedim(options$functionResult))
                a_z <- options$functionResult * 0
                                        #        dim(a_d_z@data) <- c(numel(options$functionResult), a_d_z@ndd)
                                        #        a_d_z@data[,1] <- adr_reshape(as.array(Y[adji,]), safedim(options$functionResult))
                                        #        dim(a_d_z@data) <- c(a_d_z@sz, a_d_z@ndd)
                
                ares <- adf(list(a_d_z, a_z), list(dargs, arguments, options$independents))
                
                Hplane <- do.call(jacFor, ares$adj[[2]][options$independents])
#                show(list(adji = adji, adj = ares$adj, Hplane = Hplane))
                H[adji,,idir] <- Hplane
                
                if (adji == 1) {
                    J[,idir] <- jacFor(ares$f[[1]])
                    f <- ares$f[[2]]
                }
            }
        }

    }

    ndd <- globalNDD()

    do.call(adrSetLoglev, oldLL)

    list(H = drop(H), Jv = Jv, f = f, J = J)
}

adrHessian <- adrHessRev

relmaxnorm <- function(x, y) {
    if (any(is.na(x))) {
        cat('warning: relmaxnorm: left arg has NA values\n')
        show(x)
    }
    if (any(is.na(y))) {
        cat('warning: relmaxnorm: right arg has NA values\n')
        show(x)
    }
    if (any(is.nan(x)) || any(is.nan(y))) {
        return(NaN)
    }
    if (any(is.na(x)) || any(is.na(y))) {
        return(NA)
    }
    nx <- norm(abs(as.matrix(x)), 'F')
    ny <- norm(abs(as.matrix(y)), 'F')
    nd <- norm(abs(as.matrix(x-y)), 'F')
    divi <- max(nx, ny)
    if (divi == 0) divi <- 1
    nd / divi
}

bench <- function(handle, N = 3) {
    tall.0 <- Sys.time()
    z <- handle()
    z <- handle()
    timesk <- array(0, N)
    zk <- list()
    tloop.0 <- Sys.time()
    for (k in 1:N) {
        t0 <- Sys.time()
        zk[[k]] <- handle()
        t1 <- Sys.time()
        tf <- as.double(t1 - t0, units = 'secs')
        timesk[[k]] <- tf
        if (tf > 1) cat(paste('run(k=', k, 'time=', tf, 's)'))
        cat(paste('.'))
        flush(stdout())
    }
    tloop.1 <- Sys.time()
    res <- c(mean(timesk), sd(timesk))
    cat(paste('bench(R= ', N, ', mean.time=', mean(timesk), 's)\n'))
#    if (N < 10) {
#        cat(paste(paste(timesk, collapse=', '), '\n'))
#    }
    attr(res, 'bench') <- list(N = N, times = timesk, result = z, handle = handle)
    tall.1 <- Sys.time()
#    cat(paste('bench time', as.double(tloop.1 - tloop.0, units = 'secs'),
#              's, overall bench time: ', as.double(tall.1 - tall.0, units = 'secs'), 's\n'))

    dim(res) <- c(2, 1)
    rownames(res) <- c('mean', 'std')
    res
}

fullargs <- function(args, f) {
    defs <- formals(f)
#    show(list(fullargs.enter = args, defs=defs))
    fcall <- do.call(call, append(list('f'), args))
    if (is.primitive(f)) {
        ffcall <- fcall
    } else {
        ffcall <- match.call(f, fcall)
    }
    args <- as.list(ffcall)[-1]
#    show(list(fullargs.matched = args, defs=defs))
    if (!is.null(defs) && length(defs) != 0) {
        if (is.null(names(args))) {
            ann <- args
            an <- list()
        } else {
            ann <- args[names(args) == ""]
            an <- args[names(args) != ""]
        }
        hasdot <- FALSE
        ndefs <- names(defs)
#        show(list(fullargs.named.args = an, fullargs.unnamed.args = ann))
        if ('...' %in% names(defs)) {
            hasdot <- TRUE
            defs['...'] <- NULL
            ndefs <- names(defs)
#            show(list(fullargs.hasdot = args, defs=defs, ndefs=ndefs))
        }
#        if (length(ann) > 0) {
#            names(ann) <- ndefs[!ndefs %in% names(an)][1:length(ann)]
#        }
#        args <- append(append(an, ann), defs)
                                        #        if (!hasdot) {
        bn <- an
        an <- defs
        an[names(bn)] <- bn
#        if (length(an) > 0) {
#            an <- an[ndefs]
#        }
#        } else {
#            if (length(ndefs) > 0) {
#                stop('not implemented: call to ... func with additional args')
#            }
                                        #        }
#        show(list(x=ndefs))
        bydefault <- rep(list(FALSE), length(an))
        names(bydefault) <- names(an)
#        nsyms <- 0
        niter <- 0
        while (TRUE) {
#            show(list(eval.iter=niter, args = an))
            callargs <- sapply(an, function(x) { is.expression(x) || is.call(x) || is.symbol(x) })
            callargclasses <- sapply(an, function(x) { is.expression(x)*1 + is.call(x)*2 + is.symbol(x)*4 })
#            show(list(eval.callargs=callargs, classes = callargclasses))
            if (!any(callargs) || all(callargclasses[which(callargs)] == 4)) {
                break
            } else if (niter > length(callargs)+1) {
                stop('maxiter reached in arg resolver')
                break
            } else {
                for (k in which(callargs)) {
#                    show(list(resolve.k=k))
                    aname <- names(an)[[k]]
                    ceenv <- list2env(c(an))
                    if (!is.symbol(an[k])) {
                                        # get value from default
                        aval <- try(silent = TRUE, eval(an[[k]], ceenv))
                        if (!inherits(aval, 'try-error')) {
                            an[[k]] <- aval
                            bydefault[[k]] <- TRUE
                        }
                    }
#                    else {
#                        nsyms <- nsyms + 1
#                    }
                }
            }
            niter <- niter + 1
        }
#        if (nundef > 0) {
#            logMessage('fullargs:def:eval', logLInfo, paste0("Default arguments evaluated in ", niter, " iterations"))
#        }

        args <- append(an, ann)
#        show(list(fullargs.final.arglist=args))
        args <- as.list(match.call(f, as.call(append('f', args))))[-1]
#        show(list(fullargs.final.args=args, bydef.setattr=bydefault))
        
        attr(args, 'bydefault') <- bydefault
        missargs <- sapply(args, is.symbol)
        if (any(missargs)) {
#            stop(paste('Error: missing arguments without default: ', paste(names(which(missargs)), sep='', collapse=', ')))
        }
    }
#    show(list(fullargs.done = args))
    args
}
whichargs <- function(args, defs) {
    if ('...' %in% names(defs)) {
        defs[['...']] <- NULL
    }
    if (!is.null(defs) && length(defs) > 0) {
        if (is.null(names(args))) {
            ann <- args
            an <- list()
        } else {
            ann <- args[names(args) == ""]
            an <- args[names(args) != ""]
        }
        if (length(ann) > 0) {
            names(ann) <- names(defs)[!names(defs) %in% names(an)][1:length(ann)]
        }
        anames <- names(append(an, ann))
        aninds <- names(defs) %in% anames
        arginds <- (1:length(defs))[aninds]
    } else {
        arginds <- 1:length(args)
    }
    arginds
}

dformals <- function(dfun) {
    fms <- formals(dfun)
    fms[startsWith(names(fms), 'dv_')]
}

dfinds <- function(dfun, fun) {
    fms <- formals(fun)
    if ('...' %in% names(fms)) {
        fms[['...']] <- NULL
    }
    dfms <- dformals(dfun)
    if ('...' %in% names(dfms)) {
        dfms[['...']] <- NULL
    }
    dfms <- names(dfms)
    if (is.null(names(fms)) || length(fms) == 0) {
        return(TRUE)
    } else if (is.null(dfms) || length(dfms) == 0) {
        return(1:length(dfms));
    }
    if (is.null(formals(fun))) {
        1:length(dfms)
    } else {
        sapply(names(formals(fun)), function(n) paste('dv_', n, sep='', collapse='') %in% dfms)
    }
}

dgetprefix <- function(mode = 'fm') {
    if (mode == 'fm') {
        'df_'
    } else if (mode == "rm-ret") {
        'ret_'
    } else if (mode == "rm-rec") {
        'rec_'
    } else if (mode == "rm-adj") {
        'a_'
    } else {
        show(mode)
        stop('unknown code')
    }
}

dgetcode <- function(f, indeps, fname, mode = 'fm') {
    p <- dgetprefix(mode)
    paste(p, fname, sep='', collapse='')
}

whichFrames <- function(symbol, frames) {
    r <- c()
    for (k in 1:length(frames)) {
        if (!is.null(get0(symbol, envir=frames[[k]]))) {
            r <- c(r, -(length(frames) - k))
        }
    }
    r
}

dgetfun <- function(f, args, indeps, fname) {
    code <- dgetcode(f, indeps, fname)
    actlstr = actParamsStr(f, indeps)
    fname = getSource(fname)
    logMessage('dgetfun:ddef', logLInfo, paste0("Derivative '", paste(fname, sep='', collapse=''),
                                                   "' w.r.t. ", actlstr , " required -> ", code))
    dfun <- NULL
    senv <- environment(f)
    dfenv <- cenv <- adrGetOption('srcenv', .GlobalEnv)
    fndSrcFrame <- sapply(envAncList(senv), function(x) identical(x, cenv))
    if (!is.null(senv) && any(fndSrcFrame)) {
        dfenv <- senv
    }
#    sFrames <- sys.frames()
#    seqFrames <- whichFrames('dv_seq', sFrames)
#    if (length(seqFrames)) {
#        logMessage('dgetfun:sframes', logLNote, paste0("dv_seq found in frames ",
#                                                          paste(seqFrames, sep='', collapse=', '), "", collapse=''))
#        for (k in 1:length(seqFrames)) {
#            show(k)
#            show(seqFrames[[k]])
#            printEnvNames(sFrames[[length(sFrames)+seqFrames[[k]]]])
#        }
#        show('f env')
#        printEnvNames(senv)
#    }
#    printEnvNames(cenv)
    if (!is.null(cenv) && !is.null(dfun <- get0(code, envir=cenv))) {
        logMessage('dgetfun:ddef:found', logLInfo, paste0("Derivative '", code , "' for function '", fname, "' found in d scope"))
    } else if (!is.null(senv) && !is.null(dfun <- get0(code, envir=senv))) {
        logMessage('dgetfun:ddef:found', logLInfo, paste0("Derivative '", code , "' for function '", fname, "' found in f scope"))
#    } else if (!is.null(dfun <- get0(code, envir=environment()))) {
#        logMessage('dcall:ddef:found', logLInfo, paste0("Derivative '", code , "' for function '", fname, "' found in d scope"))
#    } else if (!is.null(dfun <- get0(code, envir=globalADROptions('srcenv')))) {
#        logMessage('dcall:ddef:found', logLInfo, paste0("Derivative '", code , "' for function '", fname, "' found in s scope"))
    } else {
        logMessage('dgetfun:ddef:not:found', logLInfo, paste0("FM function '", code , "' for derivative of '",
                                                              fname, "' w.r.t. ", actlstr, " not found"))
        s <- getSource(f)
        if (length(s) > 2 && startsWith(s[[3]], 'UseMethod')) {
            cname <- class(args[[1]])
            mname <- eval(parse(text=substring("UseMethod(\"metrop\")", 10)))
            if (isS4(args[[1]])) {
                logMessage('dgetfun:S4:method', logLNote, paste0("Look up S4 method '", mname , "' for class '", cname, "'"))
                f <- getMethod(mname, cname)
            } else {
                logMessage('dgetfun:S3:method', logLNote, paste0("Look up S3 method '", mname , "' for class '", cname, "'"))
                f <- getS3method(mname, cname)
            }
        }
        dfun <- d(f, indeps, fname, envir = dfenv)
    }
    dfun
}

dcall <- function(f, dargs, args, actlist = 1:length(dargs), fname = substitute(f)) {
    fms <- formals(f)
    if (length(fms) > 0) {
        indeps <- 1:length(fms)
    } else {
        indeps <- 1:length(args)
    }
    argsinds <- whichargs(args, formals(f))
    indeps <- indeps[argsinds]
    indeps <- indeps[actlist]
    dfun <- dgetfun(f, args, indeps, fname)
    if (is.null(dfun)) {
        logMessage('dcall:dfun:null', logLError, paste0("Cannot dcall, no derivative function for '", getSource(fname), "'"))
        return(NULL)
    }

    mfargs <- fullargs(args, f)
    if (FALSE) {
    fcall <- as.call(append('f', args))
    if (is.primitive(f)) {
        ffcall <- fcall
    } else {
        ffcall <- match.call(f, fcall)
    }

    mfargs <- as.list(ffcall)[-1]
    fdef <- formals(f)
    fdef[names(mfargs)] <- NULL
    if ('...' %in% names(fdef)) {
        fdef[['...']] <- NULL
    }
    ceenv <- list2env(mfargs[names(mfargs)])

    mfargs <- append(mfargs, lapply(fdef, function(t) {
        if (!is.name(t)) {
            eval(t, ceenv)
        }
    }))
    if (!is.primitive(f)) {
        mfargs <- as.list(match.call(f, as.call(append('f', mfargs))))[-1]
    }
    }

    fcall <- as.call(append('f', dargs))
    if (is.primitive(f)) {
        ffcall <- fcall
    } else {
        ffcall <- match.call(f, fcall)
    }

    mfdargs <- as.list(ffcall)[-1]
    bydef <- attr(mfargs, 'bydefault')
    attr(mfargs, 'bydefault') <- NULL
    if (!is.null(bydef) && any(as.numeric(bydef) != 0)) {
        mfdargs <- append(mfdargs, lapply(bydef, function(t) { NULL }))
    }

    dfallargs <- list(mfdargs, mfargs)
    if ('actlist' %in% names(formals(dfun))) {
        dfallargs <- append(dfallargs, list(actlist = actlist))
    }

    logMessage('dcall:fullargs', logLDebug, paste0("Derivative call args ", paste(names(dfallargs), dfallargs, sep='=', collapse=', ')))
    do.call(dfun, dfallargs)
}


agetfun <- function(f, args, indeps, fname, adjmode="adj") {
    code <- dgetcode(f, indeps, fname, paste('rm', adjmode, collapse='', sep='-'))
    actlstr = actParamsStr(f, indeps)
    fname = getSource(fname)
    logMessage('agetfun:adef', logLInfo, paste0("Adjoint '", paste(fname, sep='', collapse=''), "', ", adjmode,
                                                   " w.r.t. ", actlstr , " required -> ", code))
    dfun <- NULL
    senv <- environment(f)
    dfenv <- cenv <- adrGetOption('srcenv', .GlobalEnv)
    fndSrcFrame <- sapply(envAncList(senv), function(x) identical(x, cenv))
    if (!is.null(senv) && any(fndSrcFrame)) {
        dfenv <- senv
    }
    if (!is.null(cenv) && !is.null(dfun <- get0(code, envir=cenv))) {
        logMessage('agetfun:adef:found', logLInfo,
                   paste0("Adjoint ", adjmode, " '", code , "' for function '", fname, "' found in d scope"))
    } else if (!is.null(senv) && !is.null(dfun <- get0(code, envir=senv))) {
        logMessage('agetfun:adef:found', logLInfo,
                   paste0("Adjoint ", adjmode, " '", code , "' for function '", fname, "' found in f scope"))
    } else {
        logMessage('agetfun:adef:not:found', logLInfo,
                   paste0("Adjoint ", adjmode, " '", code , "' for function '", fname, "' w.r.t. ", actlstr, " not found"))
        s <- getSource(f)
        if (length(s) > 2 && startsWith(s[[3]], 'UseMethod')) {
            cname <- class(args[[1]])
            mname <- eval(parse(text=substring("UseMethod(\"metrop\")", 10)))
            if (isS4(args[[1]])) {
                logMessage('dgetfun:S4:method', logLNote, paste0("Look up S4 method '", mname , "' for class '", cname, "'"))
                f <- getMethod(mname, cname)
            } else {
                logMessage('dgetfun:S3:method', logLNote, paste0("Look up S3 method '", mname , "' for class '", cname, "'"))
                f <- getS3method(mname, cname)
            }
        }
        dfun <- adj(f, indeps, fname, envir = dfenv, adjmode = adjmode)
    }
    dfun
}

acall <- function(f, dargs, args, actlist = 1:length(args), fname = substitute(f)) {
    afun <- agetfun(f, args, actlist, fname, adjmode = 'adj')
    if (is.null(afun)) {
        logMessage('dcall:afun:null', logLError, paste0("Cannot acall, no adjoint function for '", getSource(fname), "'"))
        return(NULL)
    }
    dfallargs <- list(dargs, args, actlist)
    do.call(afun, dfallargs)
}

acall.adj <- acall

acall.ret <- function(f, dargs, args, actlist = 1:length(args), fname = substitute(f)) {
    afun <- agetfun(f, args, actlist, fname, adjmode = 'ret')
    if (is.null(afun)) {
        logMessage('dcall:afun:null', logLError, paste0("Cannot acall, no adjoint function for '", getSource(fname), "'"))
        return(NULL)
    }
    fcall <- do.call(call, append(list('f'), args))
#    show(class(f))
#    show(f)
    if (is.primitive(f)) {
        ffcall <- fcall
    } else {
        ffcall <- match.call(f, fcall)
    }
#    show(as.list(fcall))
#    show(as.list(ffcall))
#    show('end')
#    lf <- length(ffcall)
    mfargs <- as.list(ffcall)[-1]
    fdef <- formals(f)
    fdef[names(mfargs)] <- NULL
    if ('...' %in% names(fdef)) {
        fdef[['...']] <- NULL
    }
    ceenv <- list2env(mfargs[names(mfargs)])
    mfargs <- append(mfargs, lapply(fdef, function(t) eval(t, ceenv)))
#    show(mfargs)
    dfallargs <- list(dargs, mfargs, actlist)
#    afcall <- do.call(call, append(list('afun'), dfallargs))
#    fafcall <- match.call(afcall, afcall)
#    show(args)
#    show(dargs)
#    show(afun)
#    show(f)
#    show(as.list(ffcall))
#    show(list(acall.ret.dfallargs = dfallargs))
#    eval(fafcall)
    do.call(afun, dfallargs)
}

acall.rec <- function(f, args, actlist = 1:length(args), fname = substitute(f)) {
    afun <- agetfun(f, args, actlist, fname, adjmode = 'rec')
    if (is.null(afun)) {
        logMessage('dcall:afun:null', logLError, paste0("Cannot acall, no adjoint function for '", getSource(fname), "'"))
        return(NULL)
    }
    dfallargs <- args
    do.call(afun, dfallargs)
}

dlist <- function(...) list(...)

#adrSetADFunName <- function(x, name = substitute(x)) {
#    skey <- getTransformCacheKey(x, 'fm', actlist, params, fname)
#    diffFunction('cache')[
#    sprintf('dv_%s', name)
#}

adrSetADFunName <- function(x, name = substitute(x)) {
    cc <- diffFunction(mode='cache')
    skey <- getTransformCacheKeyP(x, 'fm', c(), list(), fname = name)
    setFMKey <- parse(text=sprintf('df_%s', paste(name)))
    cc[[1]][[skey]] <- setFMKey
    skey <- getTransformCacheKeyP(x, 'rm', c(), list(`adjoint-mode` = 'adj'), fname = name)
    cc[[1]][[skey]] <- parse(text=sprintf('a_%s', paste(name)))
    skey <- getTransformCacheKeyP(x, 'rm', c(), list(`adjoint-mode` = 'rec'), fname = name)
    cc[[1]][[skey]] <- parse(text=sprintf('rec_%s', paste(name)))
    skey <- getTransformCacheKeyP(x, 'rm', c(), list(`adjoint-mode` = 'ret'), fname = name)
    cc[[1]][[skey]] <- parse(text=sprintf('ret_%s', paste(name)))
    diffFunction(mode='set-cache', cc)
}

adrSetADFunName(array)
#show(diffFunction(array, c(1,2), 'array', 'fm', list()))
